#!/usr/bin/env python3
"""
Wadsworth AI PCs — eBay Listing Generator

Generates formatted eBay listings from hardware info (QC JSON or interactive input).

Usage:
    python3 generate_listing.py [OPTIONS]

Options:
    --json <file>     Read hardware info from JSON file (from run_qc.sh --json)
    --price <amount>  Override price (default: based on specs)
    --output <file>   Save to file instead of stdout
    --help            Show this help message

Examples:
    python3 generate_listing.py                           # Interactive mode
    python3 generate_listing.py --json qc_output.json     # From QC JSON
    python3 generate_listing.py --json qc.json --price 199 --output listing.md
"""

import argparse
import json
import os
import sys
from pathlib import Path


# Constants
EBAY_TITLE_MAX_LENGTH = 80
DEFAULT_PRICE = 199
SCRIPT_DIR = Path(__file__).parent.resolve()
TEMPLATE_PATH = SCRIPT_DIR.parent / "templates" / "ebay_listing_template.md"


def load_template() -> str:
    """Load the eBay listing template."""
    if not TEMPLATE_PATH.exists():
        print(f"Warning: Template not found at {TEMPLATE_PATH}", file=sys.stderr)
        print("Using built-in template.", file=sys.stderr)
        return get_builtin_template()

    with open(TEMPLATE_PATH, "r", encoding="utf-8") as f:
        return f.read()


def get_builtin_template() -> str:
    """Fallback template if file not found."""
    return """# {{TITLE}}

**IMPORTANT: This computer runs Linux Mint (Cinnamon) — it is NOT Windows.**

## Specifications

| Component | Details |
|-----------|---------|
| **Model** | {{MODEL}} |
| **Processor** | {{PROCESSOR}} |
| **Memory** | {{RAM}} |
| **Storage** | {{STORAGE}} |
| **Display** | {{DISPLAY}} |
| **OS** | Linux Mint (NOT Windows) |

## Condition

{{CONDITION_NOTES}}

## What's Included

- {{MODEL}} Laptop + Charger
- Wadsworth AI Hub
- Pre-installed: Firefox, Chromium, LibreOffice, VLC

**Price: ${{PRICE}}**
"""


def generate_title(model: str, processor: str, ram: str, storage: str) -> str:
    """
    Generate eBay listing title.
    Format: [Model] [CPU] [RAM] [Storage] - Linux Mint AI-Ready
    Max 80 characters.
    """
    # Extract key CPU info (e.g., "i5-8265U" from "Intel Core i5-8265U @ 1.60GHz")
    cpu_short = processor
    if "i5" in processor.lower():
        cpu_short = "i5"
    elif "i7" in processor.lower():
        cpu_short = "i7"
    elif "i3" in processor.lower():
        cpu_short = "i3"

    # Clean up RAM (e.g., "16Gi" -> "16GB")
    ram_clean = ram.upper().replace("GI", "GB")
    # Handle case where it's just a number or ends with G
    if ram_clean.endswith("G") and not ram_clean.endswith("GB"):
        ram_clean = ram_clean + "B"
    if not ram_clean.endswith("GB") and not ram_clean.endswith("MB"):
        ram_clean = ram_clean + "GB" if ram_clean.replace(".", "").isdigit() else ram_clean

    # Clean up storage
    storage_clean = storage.upper()
    if "SSD" not in storage_clean and "HDD" not in storage_clean:
        storage_clean = storage_clean + " SSD"

    # Build title with priority components
    suffix = "Linux Mint AI-Ready"

    # Try full title first
    title = f"{model} {cpu_short} {ram_clean} {storage_clean} - {suffix}"

    if len(title) <= EBAY_TITLE_MAX_LENGTH:
        return title

    # If too long, try shorter version
    title = f"{model} {ram_clean} {storage_clean} - {suffix}"

    if len(title) <= EBAY_TITLE_MAX_LENGTH:
        return title

    # Last resort: just model and suffix
    title = f"{model} - {suffix}"

    return title[:EBAY_TITLE_MAX_LENGTH]


def calculate_price(ram: str, storage: str, processor: str) -> int:
    """
    Calculate suggested price based on specs.
    Base: $179-$229 range per REQUIREMENTS.md
    """
    base_price = 179

    # RAM bonus
    ram_value = ram.lower().replace("gb", "").replace("gi", "").strip()
    try:
        ram_gb = int(ram_value)
        if ram_gb >= 16:
            base_price += 30
        elif ram_gb >= 12:
            base_price += 15
    except ValueError:
        pass

    # Storage bonus
    storage_value = storage.lower()
    if "512" in storage_value or "500" in storage_value:
        base_price += 20
    elif "1tb" in storage_value or "1000" in storage_value:
        base_price += 40

    # CPU bonus
    processor_lower = processor.lower()
    if "i7" in processor_lower:
        base_price += 20
    elif "i5" in processor_lower and ("10th" in processor_lower or "11th" in processor_lower):
        base_price += 10

    return base_price


def prompt_for_input(prompt: str, default: str = "") -> str:
    """Prompt user for input with optional default."""
    if default:
        user_input = input(f"{prompt} [{default}]: ").strip()
        return user_input if user_input else default
    else:
        return input(f"{prompt}: ").strip()


def get_hardware_info_interactive() -> dict:
    """Collect hardware info through interactive prompts."""
    print("\n=== Wadsworth AI PC Listing Generator ===\n")
    print("Enter the hardware information for this unit.\n")

    info = {}

    info["model"] = prompt_for_input("Model (e.g., Lenovo ThinkPad T490)")
    while not info["model"]:
        print("  Model is required.")
        info["model"] = prompt_for_input("Model")

    info["processor"] = prompt_for_input("Processor (e.g., Intel Core i5-8265U)")
    while not info["processor"]:
        print("  Processor is required.")
        info["processor"] = prompt_for_input("Processor")

    info["ram"] = prompt_for_input("RAM (e.g., 16GB)", "8GB")

    info["storage"] = prompt_for_input("Storage (e.g., 256GB SSD)", "256GB SSD")

    info["display"] = prompt_for_input("Display (e.g., 14 inch FHD)", "14 inch FHD")

    info["condition_notes"] = prompt_for_input(
        "Condition notes (optional, e.g., Grade B, minor wear on palm rest)",
        "Grade B - Good condition with minor cosmetic wear. Battery holds charge."
    )

    return info


def parse_qc_json(json_path: str) -> dict:
    """Parse hardware info from QC script JSON output."""
    try:
        with open(json_path, "r", encoding="utf-8") as f:
            data = json.load(f)
    except FileNotFoundError:
        print(f"Error: File not found: {json_path}", file=sys.stderr)
        sys.exit(2)
    except json.JSONDecodeError as e:
        print(f"Error: Invalid JSON in {json_path}: {e}", file=sys.stderr)
        sys.exit(2)

    # Extract hardware info from QC JSON structure
    hw = data.get("hardware", {})

    info = {
        "model": hw.get("model", "Unknown Model"),
        "processor": hw.get("cpu", "Unknown Processor"),
        "ram": hw.get("ram", "8GB"),
        "storage": f"{hw.get('disk_size', '256GB')} SSD",
        "display": "14 inch FHD",  # Not typically in QC output
        "condition_notes": "Grade B - Good condition. Tested and verified.",
        "hostname": hw.get("hostname", ""),
        "serial": hw.get("serial", ""),
    }

    # Clean up common issues
    if info["model"] == "Unknown" or not info["model"]:
        info["model"] = "Refurbished Business Laptop"

    if info["processor"] == "Unknown" or not info["processor"]:
        info["processor"] = "Intel Core Processor"

    return info


def render_template(template: str, variables: dict) -> str:
    """Render template with variable substitution."""
    result = template
    for key, value in variables.items():
        placeholder = "{{" + key.upper() + "}}"
        result = result.replace(placeholder, str(value))
    return result


def validate_listing(title: str, info: dict) -> list:
    """Validate listing and return warnings."""
    warnings = []

    if len(title) > EBAY_TITLE_MAX_LENGTH:
        warnings.append(f"Title exceeds {EBAY_TITLE_MAX_LENGTH} chars ({len(title)} chars)")

    required_fields = ["model", "processor", "ram", "storage"]
    for field in required_fields:
        if not info.get(field) or info[field] in ["Unknown", "Unknown Model", "Unknown Processor"]:
            warnings.append(f"Missing or unknown value for: {field}")

    return warnings


def main():
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Generate eBay listings for Wadsworth AI PCs",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
    %(prog)s                              # Interactive mode
    %(prog)s --json qc_output.json        # From QC JSON
    %(prog)s --json qc.json --price 199   # With price override
    %(prog)s --output listing.md          # Save to file

The script uses the template at:
    src/templates/ebay_listing_template.md
"""
    )

    parser.add_argument(
        "--json",
        metavar="FILE",
        help="Read hardware info from JSON file (from run_qc.sh --json)"
    )

    parser.add_argument(
        "--price",
        type=int,
        metavar="AMOUNT",
        help="Override price (default: calculated from specs)"
    )

    parser.add_argument(
        "--output",
        metavar="FILE",
        help="Save listing to file instead of stdout"
    )

    args = parser.parse_args()

    # Get hardware info
    if args.json:
        info = parse_qc_json(args.json)
        print(f"Loaded hardware info from: {args.json}", file=sys.stderr)
    else:
        info = get_hardware_info_interactive()

    # Calculate or override price
    if args.price:
        price = args.price
    else:
        price = calculate_price(info["ram"], info["storage"], info["processor"])
        if not args.json:
            # In interactive mode, confirm price
            suggested = price
            price_input = prompt_for_input(f"Price (suggested: ${suggested})", str(suggested))
            try:
                price = int(price_input.replace("$", "").strip())
            except ValueError:
                price = suggested

    # Generate title
    title = generate_title(
        info["model"],
        info["processor"],
        info["ram"],
        info["storage"]
    )

    # Validate
    warnings = validate_listing(title, info)
    if warnings:
        print("\nWarnings:", file=sys.stderr)
        for warning in warnings:
            print(f"  - {warning}", file=sys.stderr)
        print("", file=sys.stderr)

    # Load template and render
    template = load_template()

    variables = {
        "TITLE": title,
        "MODEL": info["model"],
        "PROCESSOR": info["processor"],
        "RAM": info["ram"],
        "STORAGE": info["storage"],
        "DISPLAY": info.get("display", "14 inch"),
        "CONDITION_NOTES": info.get("condition_notes", "Good condition"),
        "PRICE": str(price),
    }

    listing = render_template(template, variables)

    # Output
    if args.output:
        output_path = Path(args.output)
        with open(output_path, "w", encoding="utf-8") as f:
            f.write(listing)
        print(f"\nListing saved to: {output_path}", file=sys.stderr)
        print(f"Title ({len(title)} chars): {title}", file=sys.stderr)
        print(f"Price: ${price}", file=sys.stderr)
    else:
        # Print summary to stderr, listing to stdout
        print(f"\n--- Generated Listing ---", file=sys.stderr)
        print(f"Title ({len(title)} chars): {title}", file=sys.stderr)
        print(f"Price: ${price}", file=sys.stderr)
        print(f"---\n", file=sys.stderr)
        print(listing)

    return 0


if __name__ == "__main__":
    sys.exit(main())
